/* 
 * Copyright 2015 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.fundamentals.activities;

import android.os.Bundle;
import android.support.v4.app.FragmentActivity;
import android.support.v4.view.ViewPager;
import android.view.View;
import android.view.Window;
import android.widget.Button;
import com.squareup.otto.Bus;
import de.avm.fundamentals.R;
import de.avm.fundamentals.adapter.IntroPageAdapter;
import de.avm.fundamentals.events.OnIntroPageSelectedEvent;
import de.avm.fundamentals.fragments.IntroFragment;
import de.avm.fundamentals.utils.EventBusProvider;
import de.avm.fundamentals.views.CirclePageIndicator;

public abstract class IntroActivity extends FragmentActivity implements ViewPager.OnPageChangeListener{

    protected ViewPager mViewPager;
    protected CirclePageIndicator mIndicator;
    protected IntroPageAdapter mIntroAdapter;
    protected Button mSkipButton;
    protected Button mFinishedButton;
    protected Button mNextButton;
    protected Bus mBusProvider;
    protected int mLastPage;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        requestWindowFeature(Window.FEATURE_NO_TITLE);
        setContentView(R.layout.intro_view_activity);

        mSkipButton = (Button) findViewById(R.id.intro_skip_button);
        mFinishedButton = (Button) findViewById(R.id.intro_finished_button);
        mNextButton = (Button) findViewById(R.id.intro_next_button);

        mViewPager = (ViewPager) findViewById(R.id.intro_pager);
        mViewPager.setOffscreenPageLimit(3);
        mIntroAdapter = new IntroPageAdapter(this, getSupportFragmentManager());
        initPages(mIntroAdapter);
        mViewPager.setAdapter(mIntroAdapter);

        mIndicator = (CirclePageIndicator) findViewById(R.id.indicator);
        mIndicator.setViewPager(mViewPager);
        mIndicator.setOnPageChangeListener(this);
        setButtonVisibility(false);

        mBusProvider = EventBusProvider.getInstance();
    }

    protected abstract void initPages(IntroPageAdapter adapter);

    @Override
    protected void onResume() {
        super.onResume();
        EventBusProvider.getInstance().register(this);
        animateCurrentPageFragmentIn();
    }

    @Override
    public void onPause() {
        super.onPause();
        EventBusProvider.getInstance().unregister(this);
    }

    private void animateCurrentPageFragmentIn() {
        IntroFragment fragment = (IntroFragment) getSupportFragmentManager().findFragmentByTag("android:switcher:"+ R.id.intro_pager+":"+ mViewPager.getCurrentItem());
        if(fragment != null) {
            fragment.setAnimateIntroIn(true);
        }
    }

    protected Bundle getIntroPageBundle(final int imageResId, final int textResId, final int titleResId, final int colorFromResId, final int colorToresId) {
        Bundle args = new Bundle();
        args.putInt(IntroFragment.BUNDLE_POSITION, mIntroAdapter.getCount());
        args.putInt(IntroFragment.BUNDLE_IMAGE_RES_ID, imageResId);
        args.putInt(IntroFragment.BUNDLE_MESSAGE_RES_ID, textResId);
        args.putInt(IntroFragment.BUNDLE_TITLE_RES_ID, titleResId);
        args.putInt(IntroFragment.BUNDLE_COLOR_FROM, colorFromResId);
        args.putInt(IntroFragment.BUNDLE_COLOR_TO, colorToresId);
        return args;
    }

    @Override
    public void onPageScrolled(int position, float positionOffset, int positionOffsetPixels) {
    }

    protected void setButtonVisibility(final boolean isLastPage){
        mFinishedButton.setVisibility(isLastPage ? View.VISIBLE : View.INVISIBLE);
        mNextButton.setVisibility(isLastPage ? View.INVISIBLE : View.VISIBLE);
        mSkipButton.setVisibility(isLastPage ? View.INVISIBLE : View.VISIBLE);
    }

    protected boolean isLastPage(int position) {
        return position == mIntroAdapter.getCount()-1;
    }

    @Override
    public void onPageSelected(int position) {
        OnIntroPageSelectedEvent.Direction direction;
        if(mLastPage > position){
            direction = OnIntroPageSelectedEvent.Direction.TO_LEFT;
        } else {
            direction = OnIntroPageSelectedEvent.Direction.TO_RIGHT;
        }
        mLastPage = position;
        mBusProvider.post(new OnIntroPageSelectedEvent(position, direction));
        setButtonVisibility(isLastPage(position));
    }

    @Override
    public void onPageScrollStateChanged(int state) {
        //noop
    }

    public void onClickSkip(View view) {
        finish();
    }

    public void onClickFinished(View view) {
        finish();
    }

    public void onClickNext(View view) {
        mViewPager.setCurrentItem(mViewPager.getCurrentItem() + 1, true);
    }
}
